#ifndef _WIN32
#include <unistd.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>

#include "libfma.h"
#include "lf_stdio.h"
#include "lf_internal.h"
#include "lf_fabric.h"
#include "lf_switch.h"
#include "lf_product_def.h"
#include "lf_switch_telnet.h"

#include "lf_wget.h" /* XXX */

#define DBG(s) 

/*
 * local functions
 */
static int lf_telnet_process_slot(FILE *fp, struct lf_enclosure *ep,
    char **defwp, int defwc);
static int lf_telnet_process_xbar(FILE *fp, struct lf_linecard *lp,
    char **defwp, int defwc);
static int lf_telnet_process_xbar_port(FILE *fp, struct lf_xbar *xp,
    char **defwp, int defwc);
static int lf_telnet_process_xcvr(FILE *fp, struct lf_linecard *lp,
    char **defwp, int defwc);
static int lf_telnet_process_temp(FILE *fp, struct lf_linecard *lp,
    char **defwp, int defwc);
static int lf_telnet_find_stanza_end(FILE *fp, char *keyword);
static int lf_telnet_note_new_linecard(FILE *fp, struct lf_enclosure *ep,
    int slot, char *product_id);

#define PARSE_VALUE(D, E, S) \
    else if (strcmp(wp[0], (S)) == 0) { (D)->new_vals.E = atoi(wp[2]); \
    DBG(("\t%s = %d\n", S, atoi(wp[2]))); }

int
lf_query_switch_telnet_1(
  lf_enclosure_t *ep)
{
  FILE *fp;
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  char *rv;
  int rc;
  int wc;

  /* open and read all switch components */
  fp = lf_telnet_query(ep->name, "get all\n");
  if (fp == NULL) {
    LF_ERROR(("opening telnet stream to %s", ep->name));
  }

  /* Make sure the output starts nicely */
  rv = lf_get_next_line(fp, buf, wp, &wc);
  if (rv == NULL) LF_ERROR(("Error getting first line from %s", ep->name));

  if (wc < 3 || strcmp(wp[0], "start") != 0 
      || strcmp(wp[1], "enclosure") != 0) {
    LF_ERROR(("Output error from %s", ep->name));
  }

  /* 
   * Read the whole file.  
   */
  while (lf_get_next_line(fp, buf, wp, &wc) != NULL) {

    /* See if we are starting something */
    if (strcmp(wp[0], "start") == 0) {
    
      /* make sure there is something to start */
      if (wc < 2) LF_ERROR(("Bad start line from %s", ep->name));

      /* slot stanza ? */
      if (strcmp(wp[1], "slot") == 0) {

	rc = lf_telnet_process_slot(fp, ep, wp, wc);
	if (rc != 0) LF_ERROR(("Error processing slot %s stanza from %s",
	    wp[2], ep->name));

      /* unrecognized start */
      } else {
	LF_ERROR(("Unknown start \"%s\" from enclosure %s", wp[1], ep->name));
      }

    /* ending something? */
    } else if (strcmp(wp[0], "end") == 0) {

      if (wc < 2) LF_ERROR(("Bad end stanza from %s", ep->name));

      /* "end enclosure" means we're all done */
      if (strcmp(wp[1], "enclosure") == 0) {
	goto all_done;
      }
    }
  }

  LF_ERROR(("Unexpected EOF parsing %s", ep->name));

 all_done:
  lf_telnet_end(fp);
  return LF_QUERY_OK;
  
 except:
  if (fp != NULL) lf_telnet_end(fp);
  return LF_QUERY_FATAL;
}

static int
lf_telnet_process_slot(
  FILE *fp,
  struct lf_enclosure *ep,
  char **defwp,
  int defwc)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  struct lf_linecard *lp;
  struct lf_linecard_def *ldp;
  struct lf_linecard_data *ldatap;
  char *rv;
  int slot;
  int rc;
  int wc;

  /* start slot */
  if (defwc < 8 || strcmp(defwp[2], "type") != 0
      || strcmp(defwp[5], "slot_label") != 0) {
    LF_ERROR(("Bad start slot line from %s", ep->name));
  }
  DBG(("processing slot type %s\n", defwp[4]));

  /* find the product info for this slot */
  ldp = lf_get_linecard_definition(defwp[4]);
  if (ldp == NULL) {
    LF_ERROR(("Error loading info for linecard type \"%s\" in %s",
	  defwp[4], ep->name));
  }

  /* If not a communication linecard, ignore it */
  if (ldp->lc_type != LF_LC_TYPE_COMM) {
    return lf_telnet_find_stanza_end(fp, defwp[1]);
  }

  /* get slot label */
  if (!isdigit(defwp[7][0])) {
    LF_ERROR(("Error: slot label not non-numeric \"%s\"", defwp[7]));
  }

  /* turn this into a slot number & get the linecard ptr */
  slot = atoi(defwp[7]);
  lp = ep->slots[slot];

  /* get data ptr */
  if (lp != NULL) {
    ldatap = lp->data;
    ldatap->seen = TRUE;

  /* If new linecard, report it as new and return */
  } else {
    DBG(("  [NEW]\n"));
    return lf_telnet_note_new_linecard(fp, ep, slot, defwp[4]);
  }

  /* process the rest of the stanza */
  while ((rv = lf_get_next_line(fp, buf, wp, &wc)) != NULL) {

    if (strcmp(wp[0], "serial_number") == 0) {
      if (strcmp(lp->serial_no, wp[2]) != 0) {
	/* XXX linecard changed */
      }

    /* any overtemps? */
    } else if (strcmp(wp[0], "overtemps") == 0) {
      ldatap->new_vals.overtempcount = atoi(wp[2]);

    /* start of some sub-stanza? */
    } else if (strcmp(wp[0], "start") == 0) {

      /* process a transceiver */
      if (strcmp(wp[1], "transceiver") == 0) {
	rc = lf_telnet_process_xcvr(fp, lp, wp, wc);
	if (rc == -1) {
	  LF_ERROR(("Error processing xcvr stanza for slot %d of %s",
		slot, ep->name));
	}

      /* process an xbar */
      } else if (strcmp(wp[1], "xbar") == 0) {
	rc = lf_telnet_process_xbar(fp, lp, wp, wc);
	if (rc == -1) {
	  LF_ERROR(("Error processing xbar stanza for slot %d of %s",
		slot, ep->name));
	}

      /* process a temperature */
      } else if (strcmp(wp[1], "temperature") == 0) {
	rc = lf_telnet_process_temp(fp, lp, wp, wc);
	if (rc == -1) {
	  LF_ERROR(("Error processing temperature stanza for slot %d of %s",
		slot, ep->name));
	}

      /* unknown stanza, just look for the end */
      } else {
	rc = lf_telnet_find_stanza_end(fp, wp[1]);
	if (rc == -1) {
	  LF_ERROR(("Error finding end of \"%s\" stanza, switch %s:s%d",
		wp[1], ep->name, slot));
	}
      }

    } else if (strcmp(wp[0], "end") == 0) {
      if (strcmp(wp[1], "slot") == 0) {
        goto all_done;
      } else {
        LF_ERROR(("Bad end in slot stanza: \"%s\" from %s", wp[1], ep->name));
      }
      
    }
  }
  LF_ERROR(("Unexpected EOF processing slot stanza for %s",
	ep->name));

 all_done:
  DBG(("slot done\n"));
  return 0;

 except:
  return -1;
}

static int
lf_telnet_process_xbar(
  FILE *fp,
  struct lf_linecard *lp,
  char **defwp,
  int defwc)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  struct lf_xbar *xp;
  struct lf_xbar_data *xdatap;
  char *rv;
  int x;
  int rc;
  int wc;

  /* start xbar xbar_number int 0 */
  if (defwc < 5 || strcmp(defwp[2], "xbar_number") != 0) {
    LF_ERROR(("Bad xbar slot line from %s", lp->enclosure->name));
  }

  /* get xbar number */
  x = atoi(defwp[4]);
  xp = LF_XBAR(lp->xbars[x]);
  DBG(("  start xbar %d\n", x));
  xdatap = xp->xbar_data;

  /* process the rest of the xbar stanza */
  while ((rv = lf_get_next_line(fp, buf, wp, &wc)) != NULL) {

    /* end of a stanza */
    if (strcmp(wp[0], "end") == 0) {
      if (strcmp(wp[1], "xbar") == 0) {
        goto all_done;
      } else {
        LF_ERROR(("Bad end in xbar stanza: \"%s\" from %s", wp[1],
	    lp->enclosure->name));
      }
      
    /* start of a sub-stanza */
    } else if (strcmp(wp[0], "start") == 0) {
      if (strcmp(wp[1], "xbar_port") == 0) {
	rc = lf_telnet_process_xbar_port(fp, xp, wp, wc);
	if (rc == -1) {
	  LF_ERROR(("Error processing xbar_port stanza for %s:s%d:x%d",
		xp->linecard->enclosure->name,
		lf_slot_display_no(xp->linecard), xp->xbar_no));
	}
      } else {
	LF_ERROR(("Bad start stanza \"%s\" for xbar from %s",
	      wp[1], xp->linecard->enclosure->name));
      }


    /* xbar ID */
    } else if (strcmp(wp[0], "id") == 0) {

      xp->xbar_id = atoi(wp[2]);
if (xp->xbar_id == 0) xp->xbar_id = atoi(lp->serial_no)*16 + xp->xbar_no;

    /* xbar control value */
    } else if (strcmp(wp[0], "control") == 0) {
      int control; 

      control = atoi(wp[3]);
      if (control & lp->def->qd_low_bit) {
	xdatap->new_vals.quadrant_disable |= 1;
      } else {
	xdatap->new_vals.quadrant_disable &= ~1;
      }

      if (control & lp->def->qd_high_bit) {
	xdatap->new_vals.quadrant_disable |= 2;
      } else {
	xdatap->new_vals.quadrant_disable &= ~2;
      }

    } else {
      /* ignore other keywords */
    }
  }
  LF_ERROR(("Unexpected EOF processing xbar stanza for %s",
	lp->enclosure->name));

 all_done:
  DBG(("  end xbar %d\n", xp->xbar_no));
  return 0;

 except:
  return -1;
}

static int
lf_telnet_process_xbar_port(
  FILE *fp,
  struct lf_xbar *xp,
  char **defwp,
  int defwc)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  struct lf_xbarport_data *xdp;
  char *rv;
  int port;
  int wc;

  /* start xbar_port xbar_port_number int 0 */
#if 0
  if (defwc < 5 || strcmp(defwp[2], "xbar_port_number") != 0) {
    LF_ERROR(("Bad xbar_port start line from %s",
	  xp->linecard->enclosure->name));
  }

  /* get xbar port number */
  port = atoi(defwp[4]);
#else
  port = 0;
#endif
  xdp = xp->data + port;
  DBG(("    start xbar_port %d\n", port));

  /* process the rest of the xbar stanza */
  while ((rv = lf_get_next_line(fp, buf, wp, &wc)) != NULL) {

    /* end of a sub-stanza */
    if (strcmp(wp[0], "end") == 0) {
      if (strcmp(wp[1], "xbar_port") == 0) {
        goto all_done;
      } else {
        LF_ERROR(("Bad end in xbar_port stanza for %s:s%d:x%d:p%d",
		xp->linecard->enclosure->name,
		lf_slot_display_no(xp->linecard), xp->xbar_no, port));
      }
      
    /* start of a sub-stanza */
    } else if (strcmp(wp[0], "start") == 0) {
      LF_ERROR(("Bad start stanza \"%s\" for xbar for %s:s%d:x%d:p%d",
		wp[1], xp->linecard->enclosure->name,
		lf_slot_display_no(xp->linecard), xp->xbar_no, port));
    }

    PARSE_VALUE(xdp, goodcrcs, "good_crcs")
    PARSE_VALUE(xdp, badcrcs, "bad_crcs")
    PARSE_VALUE(xdp, invalidroutes, "invalid_heads")
    PARSE_VALUE(xdp, transmittimeoutcount, "transmit_timeout_count")
    PARSE_VALUE(xdp, receivetimeoutcount, "receive_timeout_count")
    PARSE_VALUE(xdp, portdown, "port_down")
    PARSE_VALUE(xdp, portflipcount, "port_down_count")

  }
  LF_ERROR(("Unexpected EOF processing xbar_port stanza for %s:s%d:x%d:p%d",
		wp[1], xp->linecard->enclosure->name,
		lf_slot_display_no(xp->linecard), xp->xbar_no, port));

 all_done:
  DBG(("    end xbar_port %d\n", port));
  return 0;

 except:
  return -1;
}

/*
 * get a temperature value
 */
static int
lf_telnet_process_temp(
  FILE *fp,
  struct lf_linecard *lp,
  char **defwp,
  int defwc)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  char *rv;
  int wc;
  int val;
  int index;

  index = -1;		/* index unknown */
  val = 0;

  /* process the rest of the xcvr stanza */
  while ((rv = lf_get_next_line(fp, buf, wp, &wc)) != NULL) {

    /* end of a sub-stanza */
    if (strcmp(wp[0], "end") == 0) {
      if (strcmp(wp[1], "temperature") == 0) {
        goto all_done;
      } else {
        LF_ERROR(("Bad end in temperature stanza for %s:s%d",
		lp->enclosure->name, lf_slot_display_no(lp)));
      }
      
    /* start of a sub-stanza */
    } else if (strcmp(wp[0], "start") == 0) {
      LF_ERROR(("Bad start stanza \"%s\" for temp for %s:s%d:p%d",
		wp[1], lp->enclosure->name, lf_slot_display_no(lp)));

    /* which temperature */
    } else if (strcmp(wp[0], "temperature_number") == 0) {
      index = atoi(wp[2]);

    /* temperature value */
    } else if (strcmp(wp[0], "temperature") == 0) {
      val = atoi(wp[2]);
    }

    /* else ignore */

  }

  LF_ERROR(("Unexpected EOF processing temperature stanza for %s",
	lp->enclosure->name));

 all_done:
  if (index == -1) {
    LF_ERROR(("Missing temperature index for %s:s%d", 
	  lp->enclosure->name, lf_slot_display_no(lp)));
  }

  /* save the temperature value */
  lp->data->new_vals.temperature[index] = val;
  DBG(("  temp[%d] = %d\n", index, val));

  return 0;

 except:
  return -1;
}


/*
 * Process an transceiver stanza
 */
static int
lf_telnet_process_xcvr(
  FILE *fp,
  struct lf_linecard *lp,
  char **defwp,
  int defwc)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  struct lf_xcvr *xcp;
  struct lf_xcvr_data *xcdp;
  char *rv;
  int port_label;
  int port;
  int wc;

  /* start transceiver port_number int 0 */
  if (defwc < 5 || strcmp(defwp[2], "port_number") != 0) {
    LF_ERROR(("Bad transceiver start line from %s", lp->enclosure->name));
  }

  /* get xcvr port number */
  port_label = atoi(defwp[4]);
  for (port=0; port<lp->num_xcvrs; ++port) {
    /* find the right port given the label */
    if (lp->xcvr_labels[port] == port_label) {
      break;
    }
  }
  if (port >= lp->num_xcvrs) {
    LF_ERROR(("bad transceiver port label %d, slot=%d, prod=%s, "
	      "upgrade switch firmware?",
      port_label, lf_slot_display_no(lp), lp->product_id));
  }
  xcp = LF_XCVR(lp->xcvrs[port]);
  xcdp = xcp->data;

  /* process the rest of the xcvr stanza */
  while ((rv = lf_get_next_line(fp, buf, wp, &wc)) != NULL) {

    /* end of a sub-stanza */
    if (strcmp(wp[0], "end") == 0) {
      if (strcmp(wp[1], "transceiver") == 0) {
        goto all_done;
      } else {
        LF_ERROR(("Bad end in transceiver stanza for %s:s%d:p%d",
		lp->enclosure->name, lf_slot_display_no(lp), port));
      }
      
    /* start of a sub-stanza */
    } else if (strcmp(wp[0], "start") == 0) {
      LF_ERROR(("Bad start stanza \"%s\" for xbar for %s:s%d:p%d",
		wp[1], lp->enclosure->name, lf_slot_display_no(lp), port));
    }

    PARSE_VALUE(xcdp, signallost, "signal_lost")
    PARSE_VALUE(xcdp, signallostcount, "signal_lost_count")

  }
  LF_ERROR(("Unexpected EOF processing xcvr stanza for %s",
	lp->enclosure->name));

 all_done:
  return 0;

 except:
  return -1;
}

/*
 * Determine the switch type and fill in switch info if it is
 * an x32 based switch.
 */
int
lf_probe_switch_telnet_1(
  struct lf_enclosure *ep)
{
  FILE *fp;
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  char *rv;
  int rc;
  int wc;

  /* init for cleanup */
  fp = NULL;

  /*
   * First, make a pass which will tell us the enclosure type
   * and whether this enclosure speaks the telnet 1 protocol
   */
  fp = lf_telnet_query(ep->name, "get enclosure\n");
  if (fp == NULL) {
    return -1;
  }

  /*
   * first line should be enclosure type
   */
  rv = lf_get_next_line(fp, buf, wp, &wc);
  if (rv == NULL || wc != 1) {
    goto except;
  }

  /* disconnect */
  lf_telnet_end(fp);
  fp = NULL;
	
  /* save the enclosure type */
  LF_DUP_STRING(ep->product_id, wp[0]);

  /* indiciate this switch speaks telnet_1 protocol */
  ep->data->data_proto = LF_SWITCH_PROTO_TELNET_1;

  /*
   * Fill in enclosure struct based on its type
   */
  rc = lf_fill_enclosure(ep);
  if (rc == -1) goto except;

  return 0;

 except:
  if (fp != NULL) lf_telnet_end(fp);
  return -1;
}

/*
 * Discard all data until we find a line like "end <keyword>"
 */
static int
lf_telnet_find_stanza_end(
  FILE *fp,
  char *keyword)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  char *rv;
  int wc;

  while ((rv = lf_get_next_line(fp, buf, wp, &wc)) != NULL) {
    if (wc >= 2 && strcmp(wp[0], "end") == 0 && strcmp(wp[1], keyword) == 0) {
      DBG(("end-of-stanza %s\n", keyword));
      return 0;
    }
  }

  return -1;
}

/*
 * We found a new linecard - find the type
 */
static int
lf_telnet_note_new_linecard(
  FILE *fp,
  struct lf_enclosure *ep,
  int slot,
  char *product_id)
{
  char *wp[LF_STRING_LEN];
  char buf[LF_STRING_LEN];
  char *serial_no;
  int wc;
  int rc;

  while (lf_wget_next_line(fp, buf, wp, &wc) != NULL) {

    if (strcmp(wp[0], "serial_number") == 0) {
      serial_no = wp[2];
      product_id = lf_product_id_alias(product_id);
      DBG((" new %s, s/n=%s\n", product_id, serial_no));
      rc = lf_note_new_linecard(ep, slot, product_id, serial_no);

      /* If that went OK, find the end of the stanza */
      if (rc == 0) {
	rc = lf_telnet_find_stanza_end(fp, "slot");
      }
      return rc;
    }
  }

  LF_ERROR(("Unexpected EOF looking for serial_number for %s:s%d",
	ep->name, slot));

 except:
  return -1;
}
